/*!---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
import { ErrorReporter } from "./error-reporter-base";
/**
 * Defines the current state of the reporter.
 * Should only need to be used by classes extending this one.
 *
 * @export
 * @enum {number}
 */
export declare enum State {
    ready = 0,
    reporting = 1,
    finishing = 2,
}
/**
 * Configuration options
 *
 * @export
 * @interface ErrorHandlerNodeConfig
 */
export interface ErrorHandlerNodeConfig {
    /**
     * The absolute path to the root of this app. Paths in the stack trace will be stripped to here.
     * If empty, paths will be stripped to the filename.
     *
     * @type {string}
     */
    appRoot: string;
    /**
     * Information about this package to be used in the processed stack trace.
     * You can JSON.Parse or require your package.json and pass it in here, or override values.
     *
     * @type {{name: string, version?: string, branch?: string}}
     */
    packageInfo: {
        name: string;
        version?: string;
        branch?: string;
    };
    /**
     * Allows you to override the exe name used in error reports.
     * This is useful if you share an exe with another component or have a genereic name like Node.exe.
     * defaults to the real exe name if omitted.
     *
     * @type {string}
     */
    exeName?: string;
    /**
     * Allows you to determine the behavior after a crash has been reported.
     * if you return true, the app will keep running (indicates you handled it and can safely continue)
     * if you return false or do not implement this callback, the app will exit after a crash.
     */
    finalCallback?: (err: Error) => boolean;
}
/**
 * An process wide error handler for Node.js based processes.
 * Should not be constructed more than once per process.
 * sets up a global exception handler and kills the process after reporting this error unless it is stopped.
 *
 * @export
 * @class ErrorHandlerNode
 * @extends {ErrorReporter}
 */
export declare class ErrorHandlerNode extends ErrorReporter {
    /**
     * The current state, which is needed to categorize asynchronous errors.
     *
     * @protected
     * @type {State}
     */
    protected currentState: State;
    protected errorType: string;
    private finalCallback;
    /**
     * Creates an instance of ErrorHandlerNode.
     *
     * @param {Options} options - configuration options
     */
    constructor(options: ErrorHandlerNodeConfig);
    private handleUncaughtMain(err);
    private finishHandling(err);
    /**
     * The way to actually really kill the app. broken into a function to avoid duplication.
     *
     * @private
     */
    private finalExit();
    /**
     * Not actually an event emitter because we need to do async things here (like disposing the telemetry session)
     * It will call each one in the order they came in, and then exit after all the promises or after 500 ms,
     * whichever comes first.
     *
     * @param {() => Promise<any>} callback
     */
    onAboutToExit(callback: () => Promise<any>): void;
    /**
     * The list of functions which return promises to be executed right before the app dies.
     * they have 500 ms to complete or it will be killed anyway.
     *
     * @private
     */
    private aboutToExitPromises;
    /**
     * calls all the registered callbacks and waits for the promises they return to settle.
     *
     * @protected
     * @returns {Promise<any>} a promise that resolves when all registered consumers have settled their promises.
     */
    protected emitAboutToExit(): Promise<any>;
    /**
     * Adds a callback that is called when an exception goes uncaught in the middle of error reporting,
     * most likely indicating one of the reporting channels threw it in an async code section.
     *
     * @param {(asyncError: Error) => void} callback
     */
    onAsyncFailure(callback: (asyncError: Error) => void): void;
    /**
     * Calls all registered callbacks
     *
     * @protected
     * @param {Error} asyncError
     * @returns {boolean}
     */
    protected emitAsyncFailure(asyncError: Error): boolean;
    /**
     * registers a callback for when a user defined callback fails.
     *
     * @param {(uncaughtException: Error) => void} callback
     */
    onUserCallbackUncaughtException(callback: (uncaughtException: Error) => void): void;
    /**
     * tells listeners that a user defined callack has failed.
     *
     * @protected
     * @param {Error} uncaughtException - the exception that the callback threw.
     */
    protected emitUserCallbackUncaughtException(uncaughtException: Error): void;
}
